function [X,map] = readtif(filename, index)
%READTIF Read an image from a TIFF file.
%   [X,MAP] = READTIF(FILENAME) reads the first image from the
%   TIFF file specified by the string variable FILENAME.  X will
%   be a 2-D uint8 array if the specified data set contains an
%   8-bit image.  It will be an M-by-N-by-3 uint8 array if the
%   specified data set contains a 24-bit image.  MAP contains the
%   colormap if present; otherwise it is empty. 
%
%   [X,MAP] = READTIF(FILENAME, 'Index', N) reads the Nth image
%   from the file.
%
%   See also IMREAD, IMWRITE, IMFINFO.

%   Steven L. Eddins, June 1996
%   Copyright 1984-2002 The MathWorks, Inc. 
%   $Revision: 1.11 $  $Date: 2003/07/17 21:14:30 $

error(nargchk(1, 2, nargin));

if (nargin < 2)
    index = 1;
end

[X,map,photo] = rtifc(filename, index);
map = double(map)/65535;

if (photo == 8)
    % TIFF image is in CIELAB format.  Issue a warning that we're
    % converting the data to ICCLAB format, and correct the a* and b*
    % values.
    
    % First, though, check to make sure we have the expected number of
    % samples per pixel.
    if (size(X,3) ~= 1) && (size(X,3) ~= 3)
        eid = 'MATLAB:imread:unexpectedCIELabSamplesPerPixel';
        msg = 'Unexpected number of samples per pixel for CIELab image.';
        error(eid,'%s',msg);
    end
    
    wid = 'MATLAB:imread:CielabConversion';
    msg = 'Converting CIELab-encoded TIFF image to ICCLab encoding.';
    warning(wid,'%s',msg);
    
    X = cielab2icclab(X);
end

