function [hdfObjects, hdfEOSObjects] = buildHDFTree(filename)
import com.mathworks.toolbox.matlab.iofun.*;
% Create a description of an HDF/HDF-EOS file and put it in a JTree.  The
% JTree is put into a JScrollPane.

%   $Revision: 1.5 $  $Date: 2002/05/22 18:59:45 $

import javax.swing.tree.*
import java.awt.*;
import javax.swing.*

hdfEOSObjects = [];
hdfObjects    = [];

%Create nodes of the tree for HDF objects
info = hdfinfo(filename);
hdfObjects = createFileRoot(info,HDFImporter.HDF);
%Build the rest of the tree
hdfObjects = buildtree(info,hdfObjects);

%Create nodes of the tree for HDF-EOS objects
info = hdfinfo(filename,'eos');
if any([isfield(info,'Grid') isfield(info,'Point') isfield(info,'Swath')])
  hdfEOSObjects = createFileRoot(info,HDFImporter.EOS);
  hdfEOSObjects = buildtree(info,hdfEOSObjects);
end
return;

%=====================================================================
function hdfObjects = createFileRoot(info,type)
import com.mathworks.toolbox.matlab.iofun.*;
[path, shortFilename, ext] = fileparts(info.Filename);
fnames = fieldnames(info);
hdfObjects = nodeFactory.createDefaultNode(info.Filename,...
					   [shortFilename ext],...
                                           type,0);
if isfield(info,'Attributes')
  
  setNodeAttributes(info.Attributes,hdfObjects);
  
end

%=====================================================================
function root = buildtree(info,root)
import com.mathworks.toolbox.matlab.iofun.*;
import javax.swing.tree.*
import javax.swing.*

fnames = fieldnames(info);

for i=1:length(fnames)
  switch fnames{i}
   case 'SDS'
    numberOfSDS = length(info.SDS);
    for j=1:numberOfSDS
      SDSnode = createSDSNode(info.SDS(j));
      %Add SDS node to tree
      root.add(SDSnode);
    end
   case 'Vdata'
    numberOfVdata = length(info.Vdata);
    for j=1:numberOfVdata
      Vdatanode = createVdataNode(info.Vdata(j));
      root.add(Vdatanode);
    end
   case 'Vgroup'
    numberOfVgroup = length(info.Vgroup);
    for j=1:numberOfVgroup
      Vgroupnode = createVgroupNode(info,j);
      root.add(Vgroupnode);
    end
   case 'Raster8'
    numberOfRaster8 = length(info.Raster8);
    for j=1:numberOfRaster8
      raster8Node = createRaster8Node(info.Raster8(j));
      root.add(raster8Node);
    end
   case 'Raster24'
    numberOfRaster24 = length(info.Raster24);
    for j=1:numberOfRaster24
      raster24Node = createRaster24Node(info.Raster24(j));
      root.add(raster24Node);
    end
   case 'Grid'
    numberOfGrid = length(info.Grid);
    for j=1:numberOfGrid
      Gridnode = createGridNode(info.Grid(j));
      root.add(Gridnode);
    end
   case 'Point'
    numberOfPoint = length(info.Point);
    for j=1:numberOfPoint
      Pointnode = createPointNode(info.Point(j));
      root.add(Pointnode);
    end
   case 'Swath'
    numberOfSwath = length(info.Swath);
    for j=1:numberOfSwath
      Swathnode = createSwathNode(info.Swath(j));
      root.add(Swathnode);
    end
  end
end
return;

%======================================================================
function SDSNode = createSDSNode(info)
import com.mathworks.toolbox.matlab.iofun.*;
%Create node for a Scientific Data Set
SDSNode = nodeFactory.createSDSNode(info.Filename,info.Name,...
                                    HDFImporter.SDSTYPE,length(info.Dims));
SDSNode.setPrecision(char(info.DataType));
%SDS Attributes
setNodeAttributes(info.Attributes,SDSNode);
%SDS Dimensions
for k=1:length(info.Dims)
  diminfo = dimInfo(info.Dims(k).Name,info.Dims(k).Size,...
                    length(info.Dims(k).Attributes));
  setNodeAttributes(info.Dims(k).Attributes,diminfo);
  SDSNode.setDimInfo(diminfo);
end

%======================================================================
function Vdatanode = createVdataNode(info)
import com.mathworks.toolbox.matlab.iofun.*;
Vdatanode = nodeFactory.createVdataNode(info.Filename,...
					info.Name,...
					HDFImporter.VDATATYPE,...
					info.NumRecords,...
					{info.Fields.Name});
Vdatanode.setAttribute(hdfResourceBundle.getString('metadata.class'),...
                       {info.Class});
Vdatanode.setAttribute(hdfResourceBundle.getString('metadata.numrecords'),...
                       {info.NumRecords});
setNodeAttributes(info.DataAttributes,Vdatanode);

%======================================================================
function Vgroupnode = createVgroupNode(info,i)
import com.mathworks.toolbox.matlab.iofun.*;
Vgroupnode = nodeFactory.createDefaultNode(info.Vgroup(i).Filename,...
					   info.Vgroup(i).Name,...
					   HDFImporter.VGROUPTYPE,...
					   0);

Vgroupnode.setAttribute(hdfResourceBundle.getString('metadata.class'),...
                        {info.Vgroup(i).Class});
%Call buildtree recursively to take care of Vgroups
Vgroupnode = buildtree(info.Vgroup(i),Vgroupnode);

%======================================================================
function raster8Node = createRaster8Node(info)
import com.mathworks.toolbox.matlab.iofun.*;
raster8Node = nodeFactory.createRasterNode(info.Filename,...
					   info.Name,...
					   HDFImporter.RASTER8TYPE,...
					   0);
raster8Node.setAttribute(hdfResourceBundle.getString('metadata.width'),...
                         {info.Width});
raster8Node.setAttribute(hdfResourceBundle.getString('metadata.height'),...
                         {info.Height});
raster8Node.setAttribute(hdfResourceBundle.getString('metadata.colormap'),...
                         {info.HasPalette});

%======================================================================
function raster24Node = createRaster24Node(info)
import com.mathworks.toolbox.matlab.iofun.*;
raster24Node = nodeFactory.createRasterNode(info.Filename,...
					       info.Name,...
					       HDFImporter.RASTER24TYPE,...
					       0);
raster24Node.setAttribute(hdfResourceBundle.getString('metadata.bitdepth'),...
                          {24});
raster24Node.setAttribute(hdfResourceBundle.getString('metadata.width'),...
                         {info.Width});
raster24Node.setAttribute(hdfResourceBundle.getString('metadata.height'),...
                         {info.Height});
raster24Node.setAttribute(hdfResourceBundle.getString('metadata.width'),...
                          {info.Interlace});

%======================================================================
function GridNode = createGridNode(info)
import com.mathworks.toolbox.matlab.iofun.*;
GridNode = nodeFactory.createGridNode(info.Filename,...
				      info.Name,...
				      HDFImporter.GRIDTYPE,...
				      0);

numberOfAttributes = length(info.Attributes);
n = getNumProjAttrs(info.Projection.ProjCode,info.Projection.ProjParam);

setNodeAttributes(info.Attributes,GridNode)

GridNode.setAttribute(hdfResourceBundle.getString('metadata.upperleft'),...
                      {num2str(info.UpperLeft)});
GridNode.setAttribute(hdfResourceBundle.getString('metadata.lowerright'),...
                      {num2str(info.LowerRight)});
GridNode.setAttribute(hdfResourceBundle.getString('metadata.rows'),...
                      {info.Rows});
GridNode.setAttribute(hdfResourceBundle.getString('metadata.columns'),...
                      {info.Columns});
GridNode.setAttribute(hdfResourceBundle.getString('metadata.projection'),...
                      {info.Projection.ProjCode});
GridNode.setAttribute(hdfResourceBundle.getString('metadata.zonecode'),...
                      {info.Projection.ZoneCode});
GridNode.setAttribute(hdfResourceBundle.getString('metadata.sphere'),...
                      {getSphereFromCode(info.Projection.SphereCode)});
GridNode.setAttribute(hdfResourceBundle.getString('metadata.projparams'),...
                      {' '});
setProjectionParams(GridNode,info.Projection.ProjCode,info.Projection.ProjParam);
GridNode.setAttribute(hdfResourceBundle.getString('metadata.origin'),...
                      {info.OriginCode});
GridNode.setAttribute(hdfResourceBundle.getString('metadata.pixregcode'),...
                      {info.PixRegCode});

numberOfFields = length(info.DataFields);
matches = findVerticalSubsets(info);
for k=1:numberOfFields
  if length(matches)>1
    if ~isempty(matches{k,2})
      m = matches(k,2:end);
    else
      m = {};
    end
  else
    m = {};
  end
  
  dimensions = {info.DataFields(k).Dims.Name};
  for i=1:length(dimensions)
    dimensions{i} = ['DIM:' dimensions{i}];
  end
  dataFieldNode(k) = nodeFactory.createGridFieldNode(info.Filename,...
                                                    info.Name,...
                                                    info.DataFields(k).Name,...
                                                    HDFImporter.GRIDFIELDTYPE,...
                                                    length(info.DataFields(k).Dims),...
                                                    [dimensions m]);
  for j=1:length(info.DataFields(k).Dims)
    diminfo = dimInfo(info.DataFields(k).Dims(j).Name,...
		      info.DataFields(k).Dims(j).Size,...
		      0);
    dataFieldNode(k).setDimInfo(diminfo);
  end
  
  if isempty(info.DataFields(k).TileDims)
    dataFieldNode(k).setAttribute(hdfResourceBundle.getString('metadata.tiledims'),...
                                  {'No Tiles'});
  else
    dataFieldNode(k).setAttribute(hdfResourceBundle.getString('metadata.tiledims'),...
                                  {num2str(info.DataFields(k).TileDims)});
  end
  
  setNodeAttributes(info.Attributes,dataFieldNode(k));
  
  dataFieldNode(k).setAttribute(hdfResourceBundle.getString('metadata.upperleft'),...
                                {num2str(info.UpperLeft)});
  dataFieldNode(k).setAttribute(hdfResourceBundle.getString('metadata.lowerright'),...
                                {num2str(info.LowerRight)});
  dataFieldNode(k).setAttribute(hdfResourceBundle.getString('metadata.rows'),...
                                {info.Rows});
  dataFieldNode(k).setAttribute(hdfResourceBundle.getString('metadata.columns'),...
                                {info.Columns});
  dataFieldNode(k).setAttribute(hdfResourceBundle.getString('metadata.projection'),...
                                {info.Projection.ProjCode});
  dataFieldNode(k).setAttribute(hdfResourceBundle.getString('metadata.zonecode'),...
                                {info.Projection.ZoneCode});
  dataFieldNode(k).setAttribute(hdfResourceBundle.getString('metadata.sphere'),...
                                {getSphereFromCode(info.Projection.SphereCode)});
  dataFieldNode(k).setAttribute(hdfResourceBundle.getString('metadata.projparams'),...
                                {' '});
  setProjectionParams(dataFieldNode(k),info.Projection.ProjCode,info.Projection.ProjParam);
  dataFieldNode(k).setAttribute(hdfResourceBundle.getString('metadata.origin'),...
                                {info.OriginCode});
  dataFieldNode(k).setAttribute(hdfResourceBundle.getString('metadata.pixregcode'),...
                                {info.PixRegCode});
  GridNode.add(dataFieldNode(k));	
end

%======================================================================
function Pointnode = createPointNode(info)
import com.mathworks.toolbox.matlab.iofun.*;
Pointnode = nodeFactory.createDefaultNode(info.Filename,...
					  info.Name,...
					  HDFImporter.POINTTYPE,...
					  0);
for k=1:length(info.Level)
  levelnodes(k) = nodeFactory.createPointNode(info.Filename,...
                                              info.Name,...
                                              info.Level(k).Name,...
					      HDFImporter.POINTLEVELTYPE,...
					      0,...
					      info.Level(k).FieldNames);
  levelnodes(k).setAttribute(hdfResourceBundle.getString('metadata.numrecords'),...
                             {info.Level(k).NumRecords});
  numberOfAttributes = length(info.Attributes);
  
  setNodeAttributes(info.Attributes,levelnodes(k));

  Pointnode.add(levelnodes(k));
end

%======================================================================
function Swathnode = createSwathNode(info)
import com.mathworks.toolbox.matlab.iofun.*;
Swathnode = nodeFactory.createSwathNode(info.Filename,...
					info.Name,...
					HDFImporter.SWATHTYPE,...
					0);
numberOfAttributes = length(info.Attributes);

pad = java.lang.String('  ');

if ~isempty(info.MapInfo)
  for i=1:length(info.MapInfo)
    Swathnode.setAttribute(hdfResourceBundle.getString('metadata.map'),...
                           {info.MapInfo(i).Map(:)'});
    Swathnode.setAttribute(pad.concat(hdfResourceBundle.getString('metadata.offset')),...
                           {info.MapInfo(i).Offset});
    Swathnode.setAttribute(pad.concat(hdfResourceBundle.getString('metadata.increment')),...
                           {info.MapInfo(i).Increment});
  end
end
if ~isempty(info.IdxMapInfo)
  for i=1:length(info.IdxMapInfo)
    Swathnode.setAttribute(hdfResourceBundle.getString('metadata.indexmap'),...
                           {info.IdxMapInfo(i).Map(:)'});
    Swathnode.setAttribute(pad.concat(hdfResourceBundle.getString('metadata.indexsize')),...
                           {info.IdxMapInfo(i).Size});
  end
end
% Global Attributes
setNodeAttributes(info.Attributes,Swathnode);

DataFields = nodeFactory.createDefaultNode(info.Filename,...
					   hdfResourceBundle.getString('node.datafield'),...
					   HDFImporter.VGROUPTYPE,...
					   0);
GeolocationFields = nodeFactory.createDefaultNode(info.Filename,...
						  hdfResourceBundle.getString('node.geofield'),...
						  HDFImporter.VGROUPTYPE,...
						  0);

matches = findVerticalSubsets(info);
for k=1:length(info.DataFields)
  if length(matches)>1
    if ~isempty(matches{k,2})
      m = matches(k,2:end);
    else
      m = {};
    end
  else
    m ={};
  end
  
  dimensions = {info.DataFields(k).Dims.Name};
  for i=1:length(dimensions)
    dimensions{i} = ['DIM:' dimensions{i}];
  end
  datafieldnodes(k) = nodeFactory.createSwathFieldNode(info.Filename,...
						  info.Name,...
						  info.DataFields(k).Name,...
						  HDFImporter.SWATHFIELDTYPE,...
						  length(info.DataFields(k).Dims),...
                                                    [dimensions m]);
  if ~isempty(info.MapInfo)
    for i=1:length(info.MapInfo)
      datafieldnodes(k).setAttribute(hdfResourceBundle.getString('metadata.map'),...
                                     {info.MapInfo(i).Map});
      datafieldnodes(k).setAttribute(pad.concat(hdfResourceBundle.getString('metadata.offset')),...
                                     {info.MapInfo(i).Offset});
      datafieldnodes(k).setAttribute(pad.concat(hdfResourceBundle.getString('metadata.increment')),...
                                     {info.MapInfo(i).Increment});
    end
  end
  
  if ~isempty(info.IdxMapInfo)
    for i=1:length(info.IdxMapInfo)
      datafieldnodes(k).setAttribute(hdfResourceBundle.getString('metadata.indexmap'),...
                                     {info.IdxMapInfo(i).Map});
      datafieldnodes(k).setAttribute(pad.concat(hdfResourceBundle.getString('metadata.indexsize')),...
                                     {info.IdxMapInfo(i).Size});
    end 
  end
  
  setNodeAttributes(info.Attributes,datafieldnodes(k));
  
  for j=1:length(info.DataFields(k).Dims)
    diminfo = dimInfo(info.DataFields(k).Dims(j).Name,...
                      info.DataFields(k).Dims(j).Size,...
                      0);
    datafieldnodes(k).setDimInfo(diminfo);
  end
  DataFields.add(datafieldnodes(k));
end

for k=1:length(info.GeolocationFields)
  if length(matches)>1
    if ~isempty(matches{k+length(info.DataFields),2})
      m = matches(k+length(info.DataFields),2:end);
    end
  else
    m = {};
  end
  dimensions = {info.GeolocationFields(k).Dims.Name};
  for i=1:length(dimensions)
    dimensions{i} = ['DIM:' dimensions{i}];
  end
  geofieldnodes(k) = nodeFactory.createSwathFieldNode(info.Filename,...
						  info.Name,...
						  info.GeolocationFields(k).Name,...
                                                    HDFImporter.SWATHFIELDTYPE,...
						  length(info.GeolocationFields(k).Dims),...
                                                    [dimensions m]);
  for j=1:length(info.GeolocationFields(k).Dims)
    diminfo = dimInfo(info.GeolocationFields(k).Dims(j).Name,...
                      info.GeolocationFields(k).Dims(j).Size,...
                      0);
    geofieldnodes(k).setDimInfo(diminfo);
  end
  
  if ~isempty(info.MapInfo)
    for i=1:length(info.MapInfo)
      geofieldnodes(k).setAttribute(hdfResourceBundle.getString('metadata.map'),...
                                    {info.MapInfo(i).Map});
      geofieldnodes(k).setAttribute(pad.concat(hdfResourceBundle.getString('metadata.offset')),...
                                    {info.MapInfo(i).Offset});
      geofieldnodes(k).setAttribute(pad.concat(hdfResourceBundle.getString('metadata.increment')),...
                                    {info.MapInfo(i).Increment});
    end
  end
  
  if ~isempty(info.IdxMapInfo)
    for i=1:length(info.IdxMapInfo)
      geofieldnodes(k).setAttribute(hdfResourceBundle.getString('metadata.indexmap'),...
                                    {info.IdxMapInfo(i).Map});
      geofieldnodes(k).setAttribute(pad.concat(hdfResourceBundle.getString('metadata.indexsize')),...
                                    {info.IdxMapInfo(i).Size});
    end
  end
  
  for i=1:numberOfAttributes
    if isnumeric(info.Attributes(i).Value)
      value = sprintf('%f',double(info.Attributes(i).Value));
    else
      value = {info.Attributes(i).Value};
    end	  
    geofieldnodes(k).setAttribute(info.Attributes(i).Name,...
                                  {value});
  end

  GeolocationFields.add(geofieldnodes(k));
end
Swathnode.add(DataFields);
Swathnode.add(GeolocationFields);

%=====================================================================
function setNodeAttributes(attributes,node)
for i=1:length(attributes)
  if isnumeric(attributes(i).Value)
    value = num2str(double(attributes(i).Value(:)'));
  else
    value = attributes(i).Value;
  end	  
  node.setAttribute(attributes(i).Name,...
                    cellstr(value));
end

%======================================================================
function s = getProjectionNameFromCode(projCode)
codes = [0 1 4 6 7 9 11 20 22 24 99];
names = {'Geographic','Universal Transverse Mercator','Lambert Conformal Conic',...
 'Polar Stereographic','Polyconic','Transverse Mercator',...
 'Lambert Azimuthal Equal Area','Hotine Oblique Mercator','Space Oblique Mercator',...
 'Interrupted Goode Homolosine','Intergerized Sinusoidal Projection'};

i = find(codes==projCode);
if ~isempty(i)
  s = names{i};
else
  s = 'unknown projection';
end
%======================================================================
function s = getSphereFromCode(sphereCode)

codes = [0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19];
names = {'Clarke 1866',
	 'Clarke 1880',
	 'Bessel',
	 'International 1967',
	 'International 1909',
	 'WGS 72',
	 'Everest',
	 'WGS 66',
	 'GRS 1980',
	 'Airy',
	 'Modified Airy',
	 'Modified Everest',
	 'WGS 84',
	 'Southeast Asia',
	 'Austrailian National',
	 'Krassovsky',
	 'Hough',
	 'Mercury 1960',
	 'Modified Mercury 1968',
	 'Spere of Radius 6370997m'};

i = find(codes==sphereCode);
if ~isempty(i)
  s = names{i};
else
  s = 'unknown sphere';
end 
	 
%======================================================================
function setProjectionParams(node,proj,param)
import com.mathworks.toolbox.matlab.iofun.*;
if strcmp(proj,'geo')
end
if strcmp(proj,'utm')   
  node.setAttribute('    Lon/Z',{param(1)});
  node.setAttribute('    Lat/Z',{param(2)});
end
if strcmp(proj,'lamcc') 
  node.setAttribute('    Semi-Major Axis',{param(1)});
  if param(2)>0
    node.setAttribute('    Semi-Minor Axis',{param(2)});
  else
    node.setAttribute('    Ecentricity squared',{-param(2)});
  end
  node.setAttribute('    First Standard Parallel Lat',{param(3)});
  node.setAttribute('    Second Standard Parallel Lat',{param(4)});
  node.setAttribute('    Central Meridian',{param(5)});
  node.setAttribute('    Projection Origin Lat',{param(6)});
  node.setAttribute('    False Easting',{param(7)});
  node.setAttribute('    False Northing',{param(8)});
end
if strcmp(proj,'ps')    
  node.setAttribute('    Semi-Major Axis',{param(1)});
    if param(2)>0
    node.setAttribute('    Semi-Minor Axis',{param(2)});
  else
    node.setAttribute('    Ecentricity squared',{-param(2)});
  end
  node.setAttribute('    Lon below pole of map',{param(5)});
  node.setAttribute('    True Scale Lat',{param(6)});
  node.setAttribute('    False Easting',{param(7)});
  node.setAttribute('    False Northing',{param(8)});
end
if strcmp(proj,'polyc') 
  node.setAttribute('    Semi-Major Axis',{param(1)});
    if param(2)>0
    node.setAttribute('    Semi-Minor Axis',{param(2)});
  else
    node.setAttribute('    Ecentricity squared',{-param(2)});
  end
  node.setAttribute('    Central Meridian',{param(5)});
  node.setAttribute('    Projection Origin Lat',{param(6)});
  node.setAttribute('    False Easting',{param(7)});
  node.setAttribute('    False Northing',{param(8)});
end
if strcmp(proj,'tm')    
  node.setAttribute('    Semi-Major Axis',{param(1)});
  if param(2)>0
    node.setAttribute('    Semi-Minor Axis',{param(2)});
  else
    node.setAttribute('    Ecentricity squared',{-param(2)});
  end
  node.setAttribute('    Scale Factor',{param(3)});
  node.setAttribute('    Central Meridian',{param(5)});
  node.setAttribute('    Projection Origin Lat',{param(6)});
  node.setAttribute('    False Easting',{param(7)});
  node.setAttribute('    False Northing',{param(8)});
end
if strcmp(proj,'lamaz') 
  node.setAttribute('    Sphere Radius',{param(1)});
  node.setAttribute('    Proj. Center Lon',{param(5)});
  node.setAttribute('    Proj. Center Lat',{param(6)});
  node.setAttribute('    False Easting',{param(7)});
  node.setAttribute('    False Northing',{param(8)});
end
if strcmp(proj,'hom')   
  node.setAttribute('    Semi-Major Axis',{param(1)});
    if param(2)>0
    node.setAttribute('    Semi-Minor Axis',{param(2)});
  else
    node.setAttribute('    Ecentricity squared',{-param(2)});
  end
  node.setAttribute('    Scale Factor',{param(3)});
  if param(13)==1 % hom B
    node.setAttribute('    Azimuth angle east of north of center line',{param(4)});
    node.setAttribute('    Long of point on Central Meridian where azimuth occurs',{param(5)});
  end
  node.setAttribute('    Projection Origin Lat',{param(6)});
  node.setAttribute('    False Easting',{param(7)});
  node.setAttribute('    False Northing',{param(8)});
  if param(13)==0 % hom A
    node.setAttribute('    Long of 1st pt. on center line',{param(9)});
    node.setAttribute('    Lat of 1st pt. on center line',{param(10)});
    node.setAttribute('    Long of 2cd pt. on center line',{param(11)});
    node.setAttribute('    Lat of 2cd pt. on center line',{param(12)});
  end
end
if strcmp(proj,'som')   
  node.setAttribute('    Semi-Major Axis',{param(1)});
  if param(2)>0
    node.setAttribute('    Semi-Minor Axis',{param(2)});
  else
    node.setAttribute('    Ecentricity squared',{-param(2)});
  end
  if param(13)==1 % som B
    node.setAttribute('    Satellite number',{param(3)});
    node.setAttribute('    Landsat path number',{param(4)});
  end
  if param(13)==0 % som A
    node.setAttribute('    Inclination of orbit at ascending node, counter-clockwise from equator',{param(4)});
    node.setAttribute('    Lon of ascending orbit at equator',{param(5)});
  end
  node.setAttribute('    False Easting',{param(7)});
  node.setAttribute('    False Northing',{param(8)});
  if param(13)==0 % som A
    node.setAttribute('    Period of satelite in minutes',{param(9)});
    node.setAttribute('    Satellite ratio to specify start & end pt. of x,y vals on earth surface',{param(10)});
    if param(11)==0
      node.setAttribute('    Path Start/End',{'Start'});
    else
      node.setAttribute('    Path Start/End',{'End'});
    end
  end
end
if strcmp(proj,'good')    
  node.setAttribute('    Sphere Radius',{param(1)});
end
if strcmp(proj,'isinus')  
  node.setAttribute('    Sphere Radius',{param(1)});
  node.setAttribute('    Central Meridian',{param(5)});
  node.setAttribute('    False Easting',{param(7)});
  node.setAttribute('    False Northing',{param(8)});
  node.setAttribute('    Number of latitudinal zones',{param(9)});
  node.setAttribute('    Right justify columns',{param(11)});
end

%======================================================================
function n = getNumProjAttrs(proj,param)
if strcmp(proj,'geo')
  n = 0;
end
if strcmp(proj,'utm')   
  n = 2;
end
if strcmp(proj,'lamcc') 
  n = 8;
end
if strcmp(proj,'ps')    
  n = 6;
end
if strcmp(proj,'polyc') 
  n = 6;
end
if strcmp(proj,'tm')    
  n = 7;
end
if strcmp(proj,'lamaz') 
  n = 5;
end
if strcmp(proj,'hom')   
  if param(13)==1 % hom B
    n = 8;
  else
    n = 10;
  end
end
if strcmp(proj,'som')   
  if param(13)==1 % som B
    n = 6;
  end
  if param(13)==0 % som A
    n = 9;
  end
end
if strcmp(proj,'good')    
  n = 1; 
end
if strcmp(proj,'isinus')  
  n = 6
end

%======================================================================
function match = findVerticalSubsets(info)
% Returns a cell array.  The cell array has as many rows as the number of 
% fields in the Swath or Grid.  The first column is the field name.  The 
% Rest of the columns are other fields that have the following
% characteristics:
%   1.  Field has 1 dimension
%   2.  Field is int16, int32, float, or double
%   3.  A common dimension name
% In addition, to qualify as a possible vertical subset, the field must
% be monotonic, but this function does not check for this.  

match = {};
if ~isempty(info.DataFields)
    datafields_rank1   = info.DataFields(find([info.DataFields.Rank]==1));
else
    datafields_rank1 = [];
end
if isfield(info,'GeolocationFields') & ~isempty(info.GeolocationFields)
    if ~isempty(datafields_rank1)
        data_vrt_fields = datafields_rank1(find(strcmp({datafields_rank1.NumberType},'int16') |...
            strcmp({datafields_rank1.NumberType},'int32') |...
            strcmp({datafields_rank1.NumberType},'float') |...
            strcmp({datafields_rank1.NumberType},'double')));
    else
        data_vrt_fields = [];
    end
    geolocfields_rank1 = info.GeolocationFields(find([info.GeolocationFields.Rank]==1));
    if ~isempty(geolocfields_rank1)
        geo_vrt_fields = geolocfields_rank1(find(strcmp({geolocfields_rank1.NumberType},'int16') |...
            strcmp({geolocfields_rank1.NumberType},'int32') |...
            strcmp({geolocfields_rank1.NumberType},'float') |...
            strcmp({geolocfields_rank1.NumberType},'double')));
    else
        geo_vrt_fields = [];
    end 
    possible_vrt_fields = [data_vrt_fields, geo_vrt_fields];
else
    if ~isempty(datafields_rank1)
        data_vrt_fields = datafields_rank1(find(strcmp({datafields_rank1.NumberType},'int16') |...
            strcmp({datafields_rank1.NumberType},'int32') |...
            strcmp({datafields_rank1.NumberType},'float') |...
            strcmp({datafields_rank1.NumberType},'double')));
    else
        data_vrt_fields = [];
    end
    possible_vrt_fields = data_vrt_fields;
end

for i=1:length(info.DataFields)
  match{i,1} = info.DataFields(i).Name;
  match{i,2} = '';
  for j=1:length(possible_vrt_fields)
    count=2;
    if(any(strcmp(possible_vrt_fields(j).Dims.Name,{info.DataFields(i).Dims.Name})) &...
       ~strcmp(possible_vrt_fields(j).Name,info.DataFields(i).Name))
      match{i,count} = possible_vrt_fields(j).Name;
      count = count+1;
    end
  end
end

if isfield(info,'GeolocationFields')
  for i=1:length(info.GeolocationFields)
    match{i+length(info.DataFields),1} = info.GeolocationFields(i).Name;
    for j=1:length(possible_vrt_fields)
      count=2;
      if(any(strcmp(possible_vrt_fields(j).Dims.Name,{info.GeolocationFields(i).Dims.Name})) &...
	 ~strcmp(possible_vrt_fields(j).Name,info.GeolocationFields(i).Name))
	match{i+length(info.DataFields),count} = possible_vrt_fields(j).Name;
	count = count+1;
      end
    end
  end
end
